import usb.core
import can

DSP_IN_EP_ADDR = 0x81
DSP_OUT_EP_ADDR = 0x02
LS_IN_EP_ADDR = 0x83
LS_OUT_EP_ADDR = 0x04
SU_IN_EP_ADDR = 0x85
SU_OUT_EP_ADDR = 0x06

AFMAXSAMPLES = 5000

LS_IN_BUFFER_SIZE = 64
SU_IN_BUFFER_SIZE = 2 * AFMAXSAMPLES + 10

CAN_CHANNEL = 'can1'
CAN_ID = 0x20

verbosity = 0

### USB Device Initialization ###

# Find ECB USB Device
usb_device = usb.core.find(idVendor=0x0451, idProduct=0x9001)
if usb_device is None:
    raise ValueError('Device not found')
usb_device.reset()
# Set the active configuration
usb_device.set_configuration()
usb_device.set_interface_altsetting(interface=0, alternate_setting=1)


### Light Source Commands ###

ls_number = 1  # Module attribute to set the Light Source number (1-5)


def ls_get_state() -> dict:
    """Get the state of the Light Source."""
    response = ls_send_command(ls_number, cmd=0x01, data=[])
    return {'error': response[0], 'supply': response[1], 'diode_supply': response[2], 'interlock': response[3]}


def ls_enable(enable: bool = True) -> dict:
    """Enable or disable the supply of the Light Source."""
    enable = 1 if enable else 0
    response = ls_send_command(ls_number, cmd=0x05, data=[enable, enable])
    return {'supply': response[0], 'diode_supply': response[1]}


def ls_disable() -> dict:
    """Disable the supply of the Light Source."""
    return ls_enable(False)


def ls_serial_command(command: str) -> str:
    """Send serial command to the Light Source."""
    response = ls_send_command(ls_number, cmd=0x07, data=[ord(c) for c in command])
    return ''.join(chr(c) for c in response)


def ls_set_digital_modulation(value: bool):
    """Set the digital modulation state of the Light Source."""
    value = 1 if value else 0
    response = ls_send_command(ls_number, cmd=0x09, data=[value])
    return {'digital_modulation': response[0]}


def ls_set_analog_modulation(value: float):
    """Set the analog modulation value of the Light Source (0-100%)."""
    if not (0 <= value <= 100):
        raise ValueError("Analog modulation value must be between 0 and 100.")
    value = int((value / 100) * 65536)
    value = min(value, 65535)
    response = ls_send_command(ls_number, cmd=0x0B, data=[(value >> 8) & 0xFF, value & 0xFF])
    return {'analog_modulation': (response[0] << 8) | response[1]}


### Scan Unit Commands ###

def su_store_sequence(number: int, sequence: list[int]):
    """Store a sequence in the Scan Unit."""
    data = [number]
    for word in sequence:
        data.append((word >> 8) & 0xFF)
        data.append(word & 0xFF)
    response = su_send_command(cmd=0x45, data=data)
    return {'sequence_number': response[0]}


def su_start_sequence(number: int, timeout_ms: int = 1000):
    """Start a sequence in the Scan Unit."""
    response = su_send_command(cmd=0x47, data=[number], response_count=2, timeout_ms=timeout_ms)
    return {'sequence_number': response[0]}


def su_set_dac_value(channel: int, value: int):
    """Set DAC value for a specific channel in the Scan Unit."""
    channel = channel - 1
    sequence = [
        0xF000 | (value & 0x0FFF),      # Data Prefix
        0xE100 | ((channel & 7) << 4),  # Select DAC channel
        # 10 ms delay as a workaround for the bulk in buffer issue in the firmware
        0x2000 | ( 10 - 1),             # Set Cycle Time 10 * 10 us = 100 us
        0x4000 | (100 - 1),             # Wait 100 cycles (10 ms)
        0x0000,                         # Stop
    ]
    su_store_sequence(number=0, sequence=sequence)
    su_start_sequence(number=0)


def su_trigger_pulse(channel: int, duration_ms: float):
    """Trigger a pulse on a specific channel in the Scan Unit."""
    trigger_mask = 1 << (channel - 1)
    cycle_time = 100 if duration_ms >= 2 else 1  # 1 ms or 0.01 ms if duration < 2 ms
    duration_cycles = max(int(duration_ms * 100 / cycle_time), 2)
    sequence = [
        0x2000 | (cycle_time - 1),         # Set Cycle Time
        0x1000 | (trigger_mask & 0x0FFF),  # Trigger ON
        0x4000 | (duration_cycles - 2),    # Wait for duration
        0x1000,                            # Trigger OFF
        0x0000,                            # Stop
    ]
    su_store_sequence(number=0, sequence=sequence)
    su_start_sequence(number=0, timeout_ms=int(duration_ms + 1000))


def su_start_af_scan(max_len: int, options: int = 0, timeout_ms: int = 1000):
    """Start an Auto Focus scan in the Scan Unit and return the scan results."""
    response = su_send_command(cmd=0x05, data=[(max_len >> 8) & 0xFF, max_len & 0xFF, options], response_count=2, timeout_ms=timeout_ms)
    return {
        'error': response[3],
        'state': response[4],
        'no_of_samples': (response[5] << 8) | response[6],
        # 'scan_samples': response[7:],
        'scan_samples': [(response[i] << 8) | response[i + 1] for i in range(7, len(response), 2)],
    }


### CAN Commands ###

def can_get_status() -> dict:
    """Get the status of the ECB."""
    response = can_send_command(cmd=0x61, data=[])
    return {'status': response[0], 'error': response[1]}


def can_set_pwm_output(output: int, switch_current: int, hold_current: int, switch_time_ms: int) -> dict:
    """Enable/Disable the selected PWM output."""
    response = can_send_command(cmd=0x6F, data=[
        output,
        switch_current,
        hold_current,
        (switch_time_ms >> 8) & 0xFF, switch_time_ms & 0xFF,
    ])
    return {
        'output': response[0],
        'switch_current': response[1],
        'hold_current': response[2],
        'switch_time_ms': (response[3] << 8) | response[4],
    }


def can_set_shaker(frequency: int) -> dict:
    """Set the shaker frequency."""
    response = can_send_command(cmd=0x77, data=[0, (frequency >> 8) & 0xFF, frequency & 0xFF])
    return {'frequency': (response[0] << 8) | response[1]}


### USB Command Functions ###

# Command:  U8 LightSourceNr, U8 Command, U8 Reserved, U8 CmdLength, Data...
# Response: U8 LightSourceNr, U8 Command, U8 Status,   U8 CmdLength, Data...

def ls_send_command(ls: int, cmd: int, data: list[int], response_count: int = 1, timeout_ms: int = 1000) -> list[int]:
    """Send command to Light Source and read response."""
    if verbosity > 0:
        print(f"LS: -> LS={ls}, CMD=0x{cmd:02X}, LEN={len(data)}, DATA={data}")
    usb_device.write(LS_OUT_EP_ADDR, [ls, cmd, 0, len(data), *data])
    response_data = []
    for _ in range(response_count):
        _ls, _cmd, _status, _length, *_data = usb_device.read(LS_IN_EP_ADDR, LS_IN_BUFFER_SIZE, timeout_ms)
        response_data.extend(_data)
        if verbosity > 0:
            print(f"LS: <- LS={_ls}, CMD=0x{_cmd:02X}, STATUS={_status}, LEN={_length}, DATA={_data}")
    return response_data


# Command:  U8 Command, U8 Reserved, U8 CmdLength, Data...
# Response: U8 Command, U8 Status,   U8 CmdLength, Data...

def su_send_command(cmd: int, data: list[int], response_count: int = 1, timeout_ms: int = 1000) -> list[int]:
    """Send command to Scan Unit and read response."""
    if verbosity > 0:
        print(f"SU: -> CMD=0x{cmd:02X}, LEN={len(data)}, DATA={data}")
    usb_device.write(SU_OUT_EP_ADDR, [cmd, 0, (len(data) >> 8 & 0xFF), (len(data) & 0xFF), *data])
    response_data = []
    for _ in range(response_count):
        _cmd, _status, _length_msb, _length_lsb, *_data = usb_device.read(SU_IN_EP_ADDR, SU_IN_BUFFER_SIZE, timeout_ms)
        _length = (_length_msb << 8) | _length_lsb
        response_data.extend(_data)
        if verbosity > 0:
            print(f"SU: <- CMD=0x{_cmd:02X}, STATUS={_status}, LEN={_length}, DATA={_data}")
    return response_data


### CAN Command Functions ###

def can_send_command(cmd: int, data: list[int]) -> list[int]:
    """Send a CAN command and return the response data."""
    with can.interface.Bus(bustype='socketcan', channel=CAN_CHANNEL, bitrate=250000) as can_bus:
        if verbosity > 0:
            print(f"CAN: -> ID=0x{CAN_ID:02X}, CMD=0x{cmd:02X}, DATA={data}")
        return_id = CAN_ID + 0x70
        msg = can.Message(arbitration_id=CAN_ID, data=[return_id, 0, cmd, *data], is_extended_id=False)
        can_bus.send(msg)
        response = can_bus.recv(timeout=1.0)
        if response is None:
            raise TimeoutError("No response received from CAN bus.")
        _status, _ticket, _cmd, *_data = response.data
        if verbosity > 0:
            print(f"CAN: <- ID=0x{response.arbitration_id:02X}, CMD=0x{_cmd:02X}, STATUS={_status}, DATA={_data}")
        return _data

